#include <msp430xG46x.h>
#include "at_flash.h"

#define BUFFER_1_WRITE 				0x84	// buffer 1 write
#define BUFFER_2_WRITE 				0x87 	// buffer 2 write
#define BUFFER_1_READ 				0x54	// buffer 1 read
#define BUFFER_2_READ 				0x56	// buffer 2 read		
#define B1_TO_MM_PAGE_PROG_WITH_ERASE 		0x83	// buffer 1 to main memory page program with built-in erase
#define B2_TO_MM_PAGE_PROG_WITH_ERASE 		0x86	// buffer 2 to main memory page program with built-in erase
#define B1_TO_MM_PAGE_PROG_WITHOUT_ERASE 	0x88	// buffer 1 to main memory page program without built-in erase
#define B2_TO_MM_PAGE_PROG_WITHOUT_ERASE 	0x89	// buffer 2 to main memory page program without built-in erase
#define MM_PAGE_PROG_THROUGH_B1 		0x82	// main memory page program through buffer 1
#define MM_PAGE_PROG_THROUGH_B2 		0x85	// main memory page program through buffer 2
#define AUTO_PAGE_REWRITE_THROUGH_B1 		0x58	// auto page rewrite through buffer 1
#define AUTO_PAGE_REWRITE_THROUGH_B2 		0x59	// auto page rewrite through buffer 2
#define MM_PAGE_TO_B1_COMP 			0x60	// main memory page compare to buffer 1
#define MM_PAGE_TO_B2_COMP 			0x61	// main memory page compare to buffer 2
#define MM_PAGE_TO_B1_XFER 			0x53	// main memory page to buffer 1 transfer
#define MM_PAGE_TO_B2_XFER 			0x55	// main memory page to buffer 2 transfer
#define	READ_STATUS_REGISTER			0xD7	// read status register
#define CONTINUOUS_ARRAY_READ			0xE8	// continuous read
#define MAIN_MEMORY_PAGE_READ                   0x52	// main page read
#define PAGE_ERASE                              0x81	// page erase

// buffer for test
unsigned char WriteBuffer[]   = { 1, 2, 3, 4, 5, 6, 5, 8, 9, 0 };
unsigned char ReadBuffer[]    = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

void InitUCA_SPI(void) {

  // master mode, data valid on rising edge, msb first
  UCA0CTL0 = UCCKPH + UCMST + UCMSB + UCSYNC;

  // clock -> SMCLK
  UCA0CTL1 = UCSSEL1;

  // Baud Rate
  UCA0BR0 = 0x80;		
  UCA0BR0 = 0x00;			

  // second functionality of pin
  P7SEL |= 0x0e;		// P7.1-3 SPI option select
  P7DIR |= 0x0b;		// P7.0 output direction
  P7OUT |= 0x01;		// P7.0 HIGH

}

void FlashEnable(void) {
    // Set P7.0 to LOW
    P7OUT &= ~0x01;
}

void FlashDisable(void) {
    // Set P7.0 to HIGH
    P7OUT |= 0x01;
}

unsigned char WriteByte(unsigned char data) {

  while ((IFG2&UCA0TXIFG)==0);	// wait while not ready / for RX
  UCA0TXBUF = data;		// write
  while ((IFG2&UCA0RXIFG)==0);	// wait for RX buffer (full)
  return (UCA0RXBUF);

}

unsigned char FlashBusy(void)
{

    unsigned char result;

    FlashEnable();

    // Send RDSR - Read Status Register opcode
    WriteByte(READ_STATUS_REGISTER);

    // Get register contents
    result = WriteByte(0x0);

    FlashDisable();

    if(result&0x80)
      return 0;
    else
      return 1;

}

void ByteToBuffer(unsigned int byte, unsigned int address) {
	
    unsigned int addr	= 0;
    //unsigned int page	= 0;

    //page = address/264;
    addr = address%264;	

    // Wait for write to complete
    while( FlashBusy() );	

    // perform to transfer

    // Set the Write Enable latch
    FlashEnable();

    WriteByte(BUFFER_1_WRITE);

    WriteByte(0x00);

    WriteByte((*((unsigned char*)&addr +1)));

    WriteByte(((unsigned char)addr));

    // Send the byte to write
    WriteByte(byte);

    FlashDisable();

    // Wait for write to complete
    while( FlashBusy() );
}

void FlashReadArray(unsigned long address, unsigned char *buffer, unsigned char length) {

    unsigned int addr	= 0;
    unsigned int page	= 0;

    page = address/264;
    addr = address%264;

    page<<=1;	

    FlashEnable();

    // Send READ opcode
    WriteByte(CONTINUOUS_ARRAY_READ);
	
    // 24 bit page and address
    WriteByte((*((unsigned char*)&page+1)));

    WriteByte((((unsigned char)page&0xFE)|(*((unsigned char*)&addr+1) & 0x01)));

    WriteByte((unsigned char)addr);

    // 32 don't care bits
    WriteByte(0x00);
    WriteByte(0x00);
    WriteByte(0x00);
    WriteByte(0x00);

    while(length--)
    {
       *buffer++ = WriteByte(0x0);
    };
	


    FlashDisable();
}


void WriteBufferToMainMemory(unsigned int address) {
	
    //unsigned int addr	= 0;
    unsigned int page	= 0;

    page = address/264;
    //addr = address%264;	

    page<<=1;

    // Wait for write to complete
    while( FlashBusy() );	

    FlashEnable();

    //command
    WriteByte(B1_TO_MM_PAGE_PROG_WITH_ERASE);

    //6 reserved + 2 high address
    WriteByte((*((unsigned char*)&page+1)));

    //7 low address bits + 1 don't care
    WriteByte(((unsigned char)page));

    //don't care 8 bits
    WriteByte(0x00);

    FlashDisable();

    // Wait for write to complete
    while( FlashBusy() );
}

void ReadBufferFromMainMemory(unsigned int address) {

  //unsigned int addr	= 0;
  unsigned int page	= 0;

  page = address/264;
  //addr = address%264;	

  page<<=1;

  // Wait for write to complete
  while( FlashBusy() );	

  FlashEnable();

  //command
  WriteByte(MM_PAGE_TO_B1_XFER);

  //6 reserved + 2 high address
  WriteByte((*((unsigned char*)&page+1)));

  //7 low address bits + 1 don't care
  WriteByte(((unsigned char)page));

  //don't care 8 bits
  WriteByte(0x00);

  FlashDisable();

  // Wait for write to complete
  while( FlashBusy() );
}


char TestATFlash(void) {

  char i;

  // Test led and flash memory
  InitUCA_SPI();

  // Write WriteBuffer
  for(i=0; i<10; i++) {
    ByteToBuffer(WriteBuffer[i], i);
  }
  WriteBufferToMainMemory(0);

  // Read ReadBuffer
  FlashReadArray(0, ReadBuffer, 10);

  // check for Correct Read
  for(i=0; i<10; i++) {
    if(WriteBuffer[i]!=ReadBuffer[i]) {
      return 0;
    }
  }

  return 1;
}

