/*	Sample program for Olimex AVR-IO-M16 with ATMEGA16 processor
 *	Using Hyper Terminal for input commands as follow:
 *	1) W - followed by 4 binary signes (if input less than 4 signs, adds a zeros)
 *	effect: Turn ON/OFF the relays respective the signes
 *	2) w - followed by 1 hex sign (if no input sign, adds a zero)
 *	effect: TURN ON/OFF the relays respective sign converted in binary
 *	3) R - effect: output 4 binary signs depending on respective INs
 *	4) r - effect: output 1 hex sign depending on respective INs
 *	5) O/o: Set all relays ON
 *	6) C/c: Set all relays OFF
 *	7) S/s:	Switch states of relays depending on respective INs until next S/s
 *	8) Send error message on any different input
 *	Compile with AVRStudio+WinAVR (gcc version 3.4.6)
 */


#define	__AVR_ATmega16__	1
#define OSCSPEED	16000000		/* in Hz */

#include "avr/io.h"

char Txt[80];
unsigned char Action = 0;

void PORT_Init()
{
	PORTA = 0b00000000;
	DDRA = 0b00000000;

	PORTB = 0b00000000;
	DDRB = 0b00001111;		//set Relay as output (Bit0:3 = 1)

	PORTC = 0b00000000;
	DDRC = 0b00000000;

	PORTD = 0b00000000;
	DDRD = 0b00000010;		//set TX as output (Bit1 = 1)
}

/*************************************** U A R T *****************************************/

void UART_Init(uint32_t Baud)
{
	unsigned int BaudRate = OSCSPEED / (16 * Baud) - 1;		//calculate BaudRate from page 145

	//set BaudRate into registers
	UBRRH = (unsigned char) BaudRate>>8;
	UBRRL = (unsigned char) BaudRate;
	

	UCSRB = UCSRB | 0b00011000;		// RXEN & TXEN enable (Bits 4, 3 = 1)


	UCSRC = UCSRC | 0b10000110;		// set Frame Format 8 bits per frame (Bit1 = 1, Bit2 = 1), 1 stop bit(Bit3 =0)
}

unsigned char UART_Receive()
{
	if (UCSRA & 0b10000000)
		return UDR;
	else
		return 0;
}

void UART_Transmit(unsigned char Data)
{
	while (!(UCSRA & 0b00100000));
	UDR = Data;
}

/*************************************** W R I T E *****************************************/

//return Length of string
unsigned char Length(char Temp[80])
{
	unsigned char L=0;
	while (Temp[L]) L++;
	return L;
}

//print text
void Print(char Text[80])
{
	unsigned char Len, i, T;
	Len = Length(Text);
	for (i=0; i<Len; i++) 
	{
		T = Text[i];
		UART_Transmit(T);
	}
	strcpy(Text, "");
}

//print text and new line
void PrintLn(char Text[80])
{
	unsigned char Len, i, T;
	Len = Length(Text);
	for (i=0; i<Len; i++) 
	{
		T = Text[i];
		UART_Transmit(T);
	}
	strcpy(Text, "");
	UART_Transmit(13);
	UART_Transmit(10);
}

//print new line
void PrintLine()
{
	UART_Transmit(13);
	UART_Transmit(10);
}

//print welcome message
void Print_Welcome()
{
	PrintLine();
	strcpy(Txt, "************************"); PrintLn(Txt);
	strcpy(Txt, "*    AVR IO CONTROL    *"); PrintLn(Txt);
	strcpy(Txt, "* (C) 2007, OLIMEX Ltd *"); PrintLn(Txt);
	strcpy(Txt, "************************"); PrintLn(Txt);
}

//print error message
void Print_Error()
{
	unsigned char T1 = 0;
	PrintLine();
	strcpy(Txt, "ERROR!");
	PrintLn(Txt);
	//strcpy(Txt, "Press any key!");
	//PrintLn(Txt);
	//while (!(T1)) T1 = UART_Receive();
	PrintLine();
}

/******************************** A C T I O N S *****************************/

//return 1-7 depending on the choosen action or 8 for inexistant input
unsigned char Command()
{
	unsigned char Temp = 0;
	while (!(Temp))
	Temp = UART_Receive();

	if (Temp == 'W') return 1;
	if (Temp == 'w') return 2;
	if (Temp == 'R') return 3;
	if (Temp == 'r') return 4;
	if ((Temp == 'C') | (Temp == 'c')) return 5;
	if ((Temp == 'O') | (Temp == 'o')) return 6;
	if ((Temp == 'S') | (Temp == 's')) return 7;
	return 8;
}

void Write(unsigned char Temp)
{
	unsigned char T=0, T1, Br, Arr[5], Bl = 1;
	for (Br=0; Br<5; Br++) Arr[Br] = 48; //null the elements of the array (Arr[Br] = '0')
	Br = 0;
	switch (Temp)
	{
		case 1:		// Input - 'W' (Binary format input)
			while ((T != 10) & (T != 13) & (Br < 5) & Bl)
			//while	(NO ENTER (T!=10,13), Length is less than 5 signs (Br<5) and no signs except '0' or '1' (Bl = 1)
			{
				T = UART_Receive();
				if ((T != 0) & (T != 10) & (T != 13) & Bl)
				{
					if ((T != 48) & (T != 49)) Bl = 0;	//check for signs != '0' or '1'
					Arr[Br] = T;
					Br++;
				}
			}
			if ((Br == 5) | (!Bl))	//check for incorrect: input Length is 5 or signs except '0' or '1'
				Print_Error();
			else
				{
					//check value of each sign and output the relays
					if (Arr[0] == 49) 
						PORTB = PORTB | 0b00000001;
					else
						PORTB = PORTB & 0b11111110;
					if (Arr[1] == 49) 
						PORTB = PORTB | 0b00000010;
					else
						PORTB = PORTB & 0b11111101;
					if (Arr[2] == 49) 
						PORTB = PORTB | 0b00000100;
					else
						PORTB = PORTB & 0b11111011;
					if (Arr[3] == 49) 
						PORTB = PORTB | 0b00001000;
					else
						PORTB = PORTB & 0b11110111;
				}
		break;
		case 2:		// Input - 'w' (Hex format input)
			while ((T != 10) & (T != 13) & (Br < 2) & Bl)
			//while	(NO ENTER (T!=10,13), Length is less than 2 signs (Br<2) and no signs except 0-9 and A-F (Bl = 1)
			{
				T = UART_Receive();
				if ((T != 0) & (T != 10) & (T != 13) & Bl)	
				{
					Br++;
					T1 = T;		//keeping value of T (T has dynamic value (ENTER))
					if ((T1 < 48) | ((T1>57) & (T1<65)) | (T1 > 70)) //if T1(T) isn't 0-9 or A-F
						Bl = 0;
				}
			}

			T = T1;		//T back its value

			if ((!Bl) | (Br == 2))	//check for incorrect input: Length>2 or signes except 0-9 or A-F
					Print_Error();
				else
					{			
						Br = 0;		//null Br
						if (T < 58) 		//T [0..9] (0 - 48, 9 - 57)
							T = T - 48;
						else				//T [10..15] (A - 65, F - 70)
							T = T - 55;

						//transmit T in binary and keep its value in array
						while (T)
						{
							Arr[Br] = T % 2 + 48;
							Br++;
							T = T / 2;
						}

						//check value of each sign and output the relays
						if (Arr[3] == 49) 
							PORTB = PORTB | 0b00000001;
						else
							PORTB = PORTB & 0b11111110;
						if (Arr[2] == 49) 
							PORTB = PORTB | 0b00000010;
						else
							PORTB = PORTB & 0b11111101;
						if (Arr[1] == 49) 
							PORTB = PORTB | 0b00000100;
						else
							PORTB = PORTB & 0b11111011;
						if (Arr[0] == 49) 
							PORTB = PORTB | 0b00001000;
						else
							PORTB = PORTB & 0b11110111;
					}
		break;
	}
}

void Read(unsigned char Temp)
{
	unsigned char T=0;
	switch (Temp)
	{
		case 1:		// Input - 'R' (Binary format output)
			//check status of each input PIN
			if (!(PINA & 0b00000100))
				UART_Transmit('1');
			else
				UART_Transmit('0');
			if (!(PINA & 0b00000010))
				UART_Transmit('1');
			else
				UART_Transmit('0');
			if (!(PINA & 0b00000001))
				UART_Transmit('1');
			else
				UART_Transmit('0');
			if (!(PIND & 0b00000100))
				UART_Transmit('1');
			else
				UART_Transmit('0');
		break;
		case 2:		// Input - 'r' (Hex format output)
			//calculating the value depending on input PIN
			if (!(PINA & 0b00000100))	T = T + 8;
			if (!(PINA & 0b00000010))	T = T + 4;
			if (!(PINA & 0b00000001))	T = T + 2;
			if (!(PIND & 0b00000100))	T = T + 1;

			if (T < 10)			//if T in [0..9] (0 - 48; 9 - 57)
				T = T + 48;
			else
				T = T + 55;		//if T in [10..15] (A - 65; F - 70)
			UART_Transmit(T);	//output value of T
		break;
	}

}

void Control(unsigned char Temp)
{
	unsigned char T;
	switch (Temp)
	{
		case 1: PORTB = PORTB & 0b11110000; break;	//turn off all leds on relays
		case 2: PORTB = PORTB | 0b00001111; break;	//turn on all leds on relays
		case 3:	//turn on led on relays depending on input PIN
				//until input of "s" or "S"
			while ((T != 's') & (T != 'S'))
			{
				T = UART_Receive();
				//check status of input PIN and turn on/off respective led
				if (PINA & 0b00000100)
					PORTB = PORTB & 0b11111110;
				else
					PORTB = PORTB | 0b00000001;
				if (PINA & 0b00000010)
					PORTB = PORTB & 0b11111101;
				else
					PORTB = PORTB | 0b00000010;
				if (PINA & 0b00000001)
					PORTB = PORTB & 0b11111011;
				else
					PORTB = PORTB | 0b00000100;
				if (PIND & 0b00000100)
					PORTB = PORTB & 0b11110111;
				else
					PORTB = PORTB | 0b00001000;
			}
			PORTB = PORTB & 0b11110000;		//turn off all relays
		break;
	}
}

/*********************************** M A I N ********************************/

int main()
{
	PORT_Init();
	UART_Init(9600);
	while (1)
	{
		Action = 0;
		Print_Welcome();
		while (!(Action))
			Action = Command();
		switch (Action)
		{
			case 1:	Write(1); break;
			case 2:	Write(2); break;
			case 3:	Read(1); break;
			case 4:	Read(2); break;
			case 5: Control(1); break;
			case 6: Control(2); break;
			case 7: Control(3); break;
			case 8:	Print_Error(); break;
		}
	}
	return 0;
}
