/*********************************************************************
 *
 *********************************************************************
 * 			OLIMEX PIC-32MX TEST
 *********************************************************************
 *
 *
 * The purpose of this example code is to demonstrate the use of
 * peripheral library macros and functions supporting the PIC32MX
 * General Purpose Timer1, General Purpose I/O Ports and UART.
 *
 * Platform: 	Olimex PIC-32MX
 *
 * Features demonstrated:
 *      - Timer configuration
 *		- UART configuration
 *		- I/O ports configuration
 *      - Interrupt configuration
 *      - ISR syntax
 *
 * Description:
 * 		- This example toggles PORTF.RF1 (LED) every half second, time 
 *      being measured by the secondary low-power RC oscillator SOSC (working at 32,768 KHz)
 *		- When button (on PORTD.RD8) pushed it turns on the LED and keeps it permanently turned on
 *		- After reset a 'Hello' message is sent through the UART (RS232) to PC
 *		- When a character is received through RS232 an echo is returned of the character+1 
 *
 * Oscillator Configuration Bit Settings:
 * 		- Oscillator Selection Bits = 	Primary Osc w/PLL (XT+HS+EC+PLL)
 *		- Primary Oscillator Config = 	HS osc mode
 *		- PLL Input Divider			=	2x Divider
 *		- PLL Multiplier			=	18x Multiplier
 *
 * Notes:
 *		- Timer1 clock 		= SOSC/PRESCALE
 *							= 32768/1 = 32.768KHz
 *		- To toggle led once every half second, PR1 of Timer1 is loaded with 16384 = 0x4000
 *
 ********************************************************************/
#include <plib.h>

// Configuration Bit settings
// SYSCLK = 72 MHz (8MHz Crystal/ FPLLIDIV * FPLLMUL / FPLLODIV)
// PBCLK = 36 MHz
// Primary Osc w/PLL (XT+,HS+,EC+PLL)
// WDT OFF
// FSOSCEN ON - this enables the SOSC to work independantly
// Other options are don't care
//
#pragma config FPLLMUL = MUL_18, FPLLIDIV = DIV_2, FPLLODIV = DIV_1, FWDTEN = OFF, FSOSCEN = ON
#pragma config POSCMOD = HS, FNOSC = PRIPLL, FPBDIV = DIV_2

#define FOSC           		72E6

#define DESIRED_BAUDRATE    	(9600)      //The desired BaudRate

int main(void)
{
	int i;
	int	pbClk;
    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    //STEP 1. Configure cache, wait states and peripheral bus clock
	// Configure the device for maximum performance.
	// This macro sets flash wait states, PBCLK divider and DRM wait states based on the specified
	// clock frequency. It also turns on the cache mode if avaialble.
	// Based on the current frequency, the PBCLK divider will be set at 1:2. This knoweldge
	// is required to correctly set UART baud rate, timer reload value and other time sensitive
	// setting.

	pbClk = SYSTEMConfigPerformance(FOSC);

    //~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // STEP 2. configure Timer 1 using external asynchronous clock, 1:1 prescale, 0x4000 period

    OpenTimer1(T1_ON | T1_SOURCE_EXT | T1_SYNC_EXT_OFF | T1_PS_1_1, 0x4000);
			//Timer1 On, External source, No synchronization, 1:1 Prescaler, Period

    // STEP 3. Set up the timer interrupt with a priority of 2

    ConfigIntTimer1(T1_INT_ON | T1_INT_PRIOR_2);

	// STEP 4. Open and configure UART2
 
	OpenUART2(UART_EN, 					// Module is ON
	UART_RX_ENABLE | UART_TX_ENABLE,	// Enable TX & RX
	pbClk/16/DESIRED_BAUDRATE-1);		// 9600 bps, 8-N-1

	// STEP 5. Configure UART2 RX Interrupt

	ConfigIntUART2(UART_INT_PR3 | UART_RX_INT_EN | UART_TX_INT_DIS);

    // STEP 6. Enable multi-vector interrupts

    INTEnableSystemMultiVectoredInt();

	// STEP 7. Configure PORTF.RF1 = output, PORTD, RD8 = input

    PORTSetPinsDigitalOut(IOPORT_F, BIT_1);
    PORTSetPinsDigitalIn(IOPORT_D, BIT_8);
    mPORTFClearBits(BIT_1);

	i = 512*512;		// Time delay 
	while(i) {i--;}

	// Send 'Hello' message through RS23

    putsUART2("\n*** OLIMEX PIC 32 TEST ***\r\n");
    putsUART2("*** Type some characters and observe echo ***\r\n");


    while(1)
	{
	    while(mPORTDRead() & 0x0100);		// Wait until button pushed 

		while(!(mPORTDRead() & 0x0100));	// Wait until button released 

		INTEnable(INT_T1, 0);			// Disable Timer1 interrupts
		PORTSetBits(IOPORT_F, BIT_1);	// Turn on LED
	}
	    
}
	//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // STEP 8. Configure the Timer 1 interrupt handler

void __ISR(_TIMER_1_VECTOR, ipl2) Timer1Handler(void)
{
    // Toggle the LED
    mPORTFToggleBits(BIT_1);

    // Clear the interrupt flag
	mT1ClearIntFlag();
}


	// STEP 9. UART2 interrupt handler
	// it is set at priority level 3
	
void __ISR(_UART2_VECTOR, ipl3) IntUart2Handler(void)
{
	// Is this an RX interrupt?
	if(mU2RXGetIntFlag())
	{
		// Clear the RX interrupt Flag
	    mU2RXClearIntFlag();

		// Echo what we just received + 1
		putcUART2(ReadUART2()+1);
	}
}
